<?php

namespace WP_Defender;

use WP_Defender\Traits\Defender_Bootstrap;
use WP_Defender\Component\Rate;

/**
 * Class Bootstrap
 *
 * @package WP_Defender
 */
class Bootstrap {
	use Defender_Bootstrap;

	/**
	 * Activation.
	 */
	public function activation_hook(): void {
		$this->activation_hook_common();
		$this->set_free_installation_timestamp();
	}

	/**
	 * Add option with plugin install date.
	 *
	 * @since 2.4
	 */
	protected function set_free_installation_timestamp(): void {
		// Let's equate the plugin installation and postponed notice dates. This will simplify future checks.
		$install_date = (int) get_site_option( Rate::SLUG_FREE_INSTALL_DATE, 0 );
		if ( 0 === $install_date ) {
			update_site_option( Rate::SLUG_FREE_INSTALL_DATE, time() );
			update_site_option( Rate::SLUG_POSTPONED_NOTICE_DATE, time() );
		} else {
			update_site_option( Rate::SLUG_POSTPONED_NOTICE_DATE, $install_date );
		}
	}

	/**
	 * Load all modules.
	 */
	public function init_modules(): void {
		$this->init_modules_common();
		$this->init_free_dashboard();
		$this->init_black_friday();
	}

	/**
	 * Load Free Dashboard module.
	 */
	public function init_free_dashboard(): void {
		$file_path = defender_path( 'extra/free-dashboard/module.php' );
		if ( file_exists( $file_path ) ) {
			require_once $file_path;

			add_filter( 'wdev_email_message_' . DEFENDER_PLUGIN_BASENAME, array( $this, 'defender_ads_message' ) );

			$screen_prefix = 'defender';
			$screen_suffix = is_multisite() ? '-network' : '';

			$free_install_date = get_site_option( Rate::SLUG_FREE_INSTALL_DATE, false );

			do_action(
				'wpmudev_register_notices',
				'defender',
				array(
					'basename'     => DEFENDER_PLUGIN_BASENAME,
					'title'        => 'Defender',
					'wp_slug'      => 'defender-security',
					'cta_email'    => __( 'Get Secure!', 'defender-security' ),
					'installed_on' => $free_install_date ?: time(),
					'screens'      => array(
						'toplevel_page_wp-defender' . $screen_suffix,
						$screen_prefix . '_page_wdf-hardener' . $screen_suffix,
						$screen_prefix . '_page_wdf-scan' . $screen_suffix,
						$screen_prefix . '_page_wdf-logging' . $screen_suffix,
						$screen_prefix . '_page_wdf-ip-lockout' . $screen_suffix,
						$screen_prefix . '_page_wdf-waf' . $screen_suffix,
						$screen_prefix . '_page_wdf-2fa' . $screen_suffix,
						$screen_prefix . '_page_wdf-advanced-tools' . $screen_suffix,
						$screen_prefix . '_page_wdf-notification' . $screen_suffix,
						$screen_prefix . '_page_wdf-setting' . $screen_suffix,
					),
				)
			);
		}
	}

	/**
	 * Load the black friday module.
	 *
	 * @return void
	 * @since 5.6.2
	*/
	public function init_black_friday(): void {
		if ( ! class_exists( '\WPMUDEV\Modules\BlackFriday\Campaign' ) ) {
			$file_path = defender_path( 'extra/wpmudev-black-friday/campaign.php' );
			if ( file_exists( $file_path ) ) {
				require_once $file_path;
				new \WPMUDEV\Modules\BlackFriday\Campaign();
			}
		}
	}

	/**
	 * @return string
	 */
	public function defender_ads_message(): string {
		return __( "You're awesome for installing Defender! Are you interested in how to make the most of this plugin? We've collected all the best security resources we know in a single email - just for users of Defender!", 'defender-security' );
	}
}
